<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Get sale ID from URL
$sale_id = $_GET['sale_id'] ?? null;
if (!$sale_id) {
    die('Sale ID is required');
}

try {
    // Get store information
    $stmt = $pdo->prepare("SELECT setting_key, setting_value, setting_binary FROM store_settings WHERE setting_group = 'store_info'");
    $stmt->execute();
    $settings = [];
    $logoData = null;
    $logoName = '';
    while ($row = $stmt->fetch()) {
        $settings[$row['setting_key']] = $row['setting_value'];
        if ($row['setting_key'] === 'store_logo') {
            $logoData = $row['setting_binary'];
            $logoName = $row['setting_value'];
        }
    }

    // Debug output to verify store settings
    error_log("Store Settings: " . print_r($settings, true));

    // Get sale details
    $stmt = $pdo->prepare("
        SELECT s.*, c.name as customer_name, c.phone as customer_phone, c.email as customer_email,
               u.username as processed_by,
               DATE_FORMAT(s.created_at, '%Y-%m-%d %H:%i:%s') as sale_date,
               s.invoice_number
        FROM sales s
        LEFT JOIN customers c ON s.customer_id = c.id
        LEFT JOIN users u ON s.user_id = u.id
        WHERE s.id = ?
    ");
    $stmt->execute([$sale_id]);
    $sale = $stmt->fetch();

    if (!$sale) {
        die('Sale not found');
    }

    // Get sale items
    $stmt = $pdo->prepare("
        SELECT si.*, p.name as product_name, c.require_serials,
               GROUP_CONCAT(psn.serial_number) as serial_numbers
        FROM sale_items si 
        JOIN products p ON si.product_id = p.id 
        LEFT JOIN categories c ON p.category_id = c.id
        LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id 
            AND (
                (psn.status = 'sold' AND psn.source_id = si.sale_id AND psn.source_type = 'manual_adjustment')
                OR 
                (psn.status = 'sold' AND psn.source_id = si.sale_id AND psn.source_type = 'credit_sale')
            )
        WHERE si.sale_id = ?
        GROUP BY si.id
    ");
    $stmt->execute([$sale_id]);
    $sale_items = $stmt->fetchAll();

    // For each sale item, fetch the serial numbers sold in this sale
    foreach ($sale_items as &$item) {
        if (!empty($item['serial_numbers'])) {
            $item['serial_numbers'] = str_replace(',', ', ', $item['serial_numbers']);
        }
    }
    unset($item);

    // Get sale payments
    $stmt = $pdo->prepare("SELECT method, amount FROM sale_payments WHERE sale_id = ?");
    $stmt->execute([$sale_id]);
    $sale_payments = $stmt->fetchAll();

    // Calculate total discount before table header
    $total_discount = 0;
    foreach ($sale_items as $item) {
        $total_discount += $item['discount'] ?? 0;
    }

    // Fetch default prices for all products in this sale
    $product_ids = array_map(function($item) { return $item['product_id']; }, $sale_items);
    $default_prices = [];
    if (!empty($product_ids)) {
        $placeholders = implode(',', array_fill(0, count($product_ids), '?'));
        $stmt = $pdo->prepare("SELECT id, price FROM products WHERE id IN ($placeholders)");
        $stmt->execute($product_ids);
        while ($row = $stmt->fetch()) {
            $default_prices[$row['id']] = $row['price'];
        }
    }
    // Calculate total
    $total = 0;
    foreach ($sale_items as $item) {
        $item_discount = $item['discount'] ?? 0;
        $subtotal = ($item['price'] * $item['quantity']) - $item_discount;
        $total += $subtotal;
    }

} catch (PDOException $e) {
    die('Error: ' . $e->getMessage());
}

// Get currency from settings
$currency = $settings['currency'] ?? 'Ksh';
$website = $settings['website'] ?? '';

// Store info for header (exclude currency and tax_rate)
$store_header = [
    'store_name' => $settings['store_name'] ?? '',
    'store_address' => $settings['store_address'] ?? '',
    'store_phone' => $settings['store_phone'] ?? '',
    'store_email' => $settings['store_email'] ?? '',
    'website' => $settings['website'] ?? '',
    'store_logo' => $settings['store_logo'] ?? ''
];

// Construct the logo path relative to the receipt file
$logo_path = '';
if (!empty($store_header['store_logo'])) {
    $logo_path = '../' . $store_header['store_logo'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sales Receipt <?php echo $sale['invoice_number']; ?></title>
    <style>
        @page {
            size: 80mm auto;
            margin: 0;
        }
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
            margin: 0;
            padding: 0;
            font-size: 12px;
            width: 80mm;
            max-width: 80mm;
            background: white;
        }
        .receipt {
            width: 80mm;
            max-width: 80mm;
            margin: 0;
            padding: 2mm;
            background: white;
            color: black;
        }
        .header {
            text-align: center;
            margin-bottom: 5px;
            padding-bottom: 3px;
            border-bottom: 1px dashed #000;
        }
        .store-logo {
            max-width: 60mm;
            max-height: 30mm;
            margin: 5px auto;
            display: block;
        }
        .store-name {
            font-size: 20px;
            font-weight: 800;
            margin: 10px 0;
            color: #000;
        }
        .store-details {
            font-size: 13px;
            margin: 5px 0;
            padding-bottom: 10px;
            border-bottom: 1px solid #000;
            color: #000;
        }
        .store-details p {
            margin: 3px 0;
            line-height: 1.4;
        }
        .receipt-header {
            text-align: center;
            margin-bottom: 5px;
            padding: 2px;
        }
        .receipt-number {
            font-size: 13px;
            font-weight: bold;
            margin-bottom: 2px;
            color: black;
        }
        .receipt-date, .client-info {
            font-size: 11px;
            margin-bottom: 1px;
            color: black;
        }
        .receipt-title {
            text-align: center;
            font-size: 22px;
            font-weight: 800;
            margin: 15px 0;
            padding: 10px 0;
            border-bottom: 2px solid #000;
            color: #000;
        }
        .sale-details {
            margin: 20px 0;
            text-align: center;
            padding: 15px 0;
        }
        .sale-details div {
            margin: 5px 0;
            font-size: 13px;
            white-space: nowrap;
            color: #000;
        }
        .sale-details .label {
            font-weight: 700;
            display: inline-block;
            width: 100px;
            text-align: right;
            margin-right: 5px;
        }
        .sale-details .value {
            display: inline-block;
            text-align: left;
            font-weight: 600;
        }
        .items {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        .items th, .items td {
            padding: 3px;
            text-align: left;
            border-bottom: 1px solid #000;
            font-size: 12px;
            color: #000;
        }
        .items th {
            font-weight: 800;
            border-bottom: 2px solid #000;
        }
        .items td {
            font-weight: 600;
        }
        .total {
            text-align: right;
            font-weight: 800;
            margin-top: 15px;
            font-size: 14px;
            color: #000;
        }
        .separator {
            border-top: 1px solid #000;
            margin: 20px 0;
        }
        .thank-you {
            text-align: center;
            font-size: 14px;
            margin: 15px 0;
            font-style: italic;
            font-weight: 600;
            color: #000;
        }
        .footer {
            text-align: center;
            margin-top: 20px;
            font-size: 12px;
            font-weight: 600;
            color: #000;
        }
        .credit {
            text-align: center;
            margin-top: 10px;
            font-size: 12px;
            color: #000;
            border-top: 1px solid #000;
            padding-top: 10px;
            font-weight: 800;
        }
        .payment-details {
            margin: 15px 0;
            text-align: center;
            font-weight: 600;
            color: #000;
        }
        .payment-details div {
            margin: 5px 0;
            font-size: 13px;
        }
        .payment-details .label {
            font-weight: 700;
            display: inline-block;
            width: 100px;
            text-align: right;
            margin-right: 5px;
        }
        .payment-details .value {
            display: inline-block;
            text-align: left;
        }
        @media print {
            html, body {
                width: 80mm;
                height: auto;
                margin: 0;
                padding: 0;
                background: white;
            }
            .receipt {
                width: 80mm;
                margin: 0;
                padding: 2mm;
                background: white;
            }
            .qr-section img {
                width: 100px;
                height: 100px;
            }
            * {
                -webkit-print-color-adjust: exact !important;
                print-color-adjust: exact !important;
                color-adjust: exact !important;
            }
            .receipt * {
                color: black !important;
                background: white !important;
            }
        }
        .dotted-line { 
            border-top: 1px dotted #000; 
            margin: 5px 0; 
        }
        .section-separator { 
            border-top: 1px solid #000; 
            margin: 5px 0; 
        }
        .thankyou { 
            text-align: center; 
            font-weight: bold; 
            font-size: 11px; 
            margin: 4px 0;
            color: black;
        }
        .qr-section { 
            text-align: center; 
            margin: 4px 0; 
        }
        .credit-footer { 
            text-align: center; 
            font-weight: bold; 
            font-size: 11px; 
            margin-top: 5px;
            color: black;
        }
    </style>
    <script>
        window.onload = function() {
            window.print();
            setTimeout(function() {
                window.close();
            }, 1000);
        }
    </script>
</head>
<body>
<div class="receipt">
    <!-- Store Header -->
    <div class="header">
        <?php if ($logoData): ?>
            <?php
            $mimeType = 'image/jpeg'; // Default to JPEG
            if (strpos($logoName, '.png') !== false) {
                $mimeType = 'image/png';
            } elseif (strpos($logoName, '.gif') !== false) {
                $mimeType = 'image/gif';
            }
            ?>
            <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" class="store-logo">
        <?php endif; ?>
        <div class="store-name"><?php echo htmlspecialchars($store_header['store_name']); ?></div>
        <div class="store-details">
                <p><?php echo htmlspecialchars($store_header['store_address']); ?></p>
            <p>Phone: <?php echo htmlspecialchars($store_header['store_phone']); ?></p>
                <p>Email: <?php echo htmlspecialchars($store_header['store_email']); ?></p>
            <?php if (!empty($store_header['website'])): ?>
                <p>Web: <?php echo htmlspecialchars($store_header['website']); ?></p>
            <?php endif; ?>
        </div>
    </div>
    <div class="receipt-header">
        <div class="receipt-number">Receipt #: <?php echo $sale['invoice_number']; ?></div>
        <div class="receipt-date">Date: <?php echo $sale['sale_date']; ?></div>
        <div class="client-info">Client: <?php echo htmlspecialchars($sale['customer_name'] ?? 'Walk-in'); ?></div>
        <div class="salesperson-info">Cashier: <?php echo htmlspecialchars($sale['processed_by'] ?? 'Unknown'); ?></div>
        <div style="border-top: 1px solid #000; margin: 8px 0 0 0;"></div>
    </div>
    <div class="receipt-title" style="margin-top: 0;">SALES RECEIPT</div>
    <table style="width: 100%; border-collapse: collapse;">
        <thead>
            <tr>
                <th style="text-align: left;">Item</th>
                <th style="text-align: center;">Qty</th>
                <th style="text-align: right;">Price</th>
                <?php if ($total_discount > 0): ?>
                <th style="text-align: right;">Discount</th>
                <?php endif; ?>
                <th style="text-align: right;">Total</th>
            </tr>
        </thead>
        <tbody>
        <?php foreach ($sale_items as $item): ?>
            <?php 
                $item_discount = $item['discount'] ?? 0;
                $subtotal = ($item['price'] * $item['quantity']) - $item_discount;
            ?>
            <tr>
                <td style="text-align: left;">
                    <?php echo htmlspecialchars($item['product_name']); ?>
                    <?php if (!empty($item['serial_numbers'])): ?>
                        <br><small>SN: <?php echo htmlspecialchars($item['serial_numbers']); ?></small>
                    <?php endif; ?>
                </td>
                <td style="text-align: center;"><?php echo $item['quantity']; ?></td>
                <td style="text-align: right;"><?php echo $currency . ' ' . number_format($item['price'], 2); ?></td>
                <?php if ($total_discount > 0): ?>
                <td style="text-align: right;"><?php echo $currency . ' ' . number_format($item_discount, 2); ?></td>
                <?php endif; ?>
                <td style="text-align: right;"><?php echo $currency . ' ' . number_format($subtotal, 2); ?></td>
            </tr>
        <?php endforeach; ?>
        </tbody>
    </table>
    <div class="section-separator"></div>
    <div class="info-row" style="display: flex; justify-content: space-between; margin: 5px 0;">
        <span class="info-label">Subtotal:</span>
        <span><?php echo $currency . ' ' . number_format($total, 2); ?></span>
    </div>
    <?php if (!empty($sale['shipping']) && $sale['shipping'] > 0): ?>
    <div class="info-row" style="display: flex; justify-content: space-between; margin: 5px 0;">
        <span class="info-label">Shipping:</span>
        <span><?php echo $currency . ' ' . number_format($sale['shipping'], 2); ?></span>
    </div>
    <?php endif; ?>
    <div class="info-row" style="display: flex; justify-content: space-between; margin: 5px 0; font-weight: bold;">
        <span class="info-label">Total Payable:</span>
        <span><?php echo $currency . ' ' . number_format($total + $sale['shipping'], 2); ?></span>
    </div>
    <div class="section-separator"></div>
    <div class="info-row">
        <span class="info-label">Amount Paid:</span>
        <span>
        <?php 
        $is_credit_sale = false;
        if (count($sale_payments) === 1 && strtolower($sale_payments[0]['method']) === 'credit') {
            $is_credit_sale = true;
        }
        $total_paid = 0;
        $shown_methods = [];
        foreach ($sale_payments as $pay) {
            $method = strtolower($pay['method']);
            if ($method === 'cash' || $method === 'mpesa') {
                $total_paid += $pay['amount'];
                $shown_methods[] = $method;
            }
        }
        // If no cash or mpesa was shown, fallback to credit
        if ($is_credit_sale) {
            // Only show credit line for pure credit sale
            echo '<div style="font-weight: bold; color: #e11d48; margin-top: 2px;">';
            echo 'Credit: ' . $currency . ' ' . number_format($sale_payments[0]['amount'], 2);
            echo '</div>';
        } else if (empty($shown_methods)) {
            foreach ($sale_payments as $pay) {
                if (strtolower($pay['method']) === 'credit') {
                    $total_paid += $pay['amount'];
                    $shown_methods[] = 'credit';
                ?>
                <ul style="margin:0; padding-left:18px;">
                    <li><?php echo htmlspecialchars(ucfirst($pay['method'])); ?>: <?php echo $currency . ' ' . number_format($pay['amount'], 2); ?></li>
                </ul>
                <?php break; }
            }
            echo '<div style="font-weight: bold; color: #e11d48; margin-top: 2px;">';
            echo 'Credit: ' . $currency . ' ' . number_format($total_paid, 2);
            echo '</div>';
        } else {
            // Show cash/mpesa lines
            echo '<ul style="margin:0; padding-left:18px;">';
            foreach ($sale_payments as $pay) {
                $method = strtolower($pay['method']);
                if ($method === 'cash' || $method === 'mpesa') {
                    echo '<li>' . htmlspecialchars(ucfirst($pay['method'])) . ': ' . $currency . ' ' . number_format($pay['amount'], 2) . '</li>';
                }
            }
            echo '</ul>';
            echo '<div style="font-weight: bold; margin-top: 5px;">';
            echo 'Total Paid: ' . $currency . ' ' . number_format($total_paid, 2);
            echo '</div>';
            // Show credit line if present
            $credit_amount = 0;
            foreach ($sale_payments as $pay) {
                if (strtolower($pay['method']) === 'credit') {
                    $credit_amount += $pay['amount'];
                }
            }
            if ($credit_amount > 0) {
                echo '<div style="font-weight: bold; color: #e11d48; margin-top: 2px;">';
                echo 'Credit: ' . $currency . ' ' . number_format($credit_amount, 2);
                echo '</div>';
            }
        }
        ?>
        </span>
    </div>
    <?php if (!empty($sale['cash_tendered'])): ?>
    <div class="info-row">
        <span class="info-label">Cash Tendered:</span>
        <span><?php echo $currency . ' ' . number_format($sale['cash_tendered'], 2); ?></span>
    </div>
    <div class="info-row">
        <span class="info-label">Change:</span>
        <span><?php echo $currency . ' ' . number_format($sale['change_amount'], 2); ?></span>
    </div>
    <?php endif; ?>
    <?php if ($sale['paid_amount'] < ($total + $sale['shipping'])): ?>
    <div class="info-row">
        <span class="info-label">Remaining:</span>
        <span><?php echo $currency . ' ' . number_format(($total + $sale['shipping']) - $sale['paid_amount'], 2); ?></span>
    </div>
    <?php endif; ?>
    <?php if ($total_discount > 0): ?>
    <div class="info-row" style="display: flex; justify-content: space-between; margin: 5px 0;">
        <span class="info-label">Total Discount:</span>
        <span><?php echo $currency . ' ' . number_format($total_discount, 2); ?></span>
    </div>
    <?php endif; ?>
    <div class="thank-you">
      Thank you for shopping with <?php echo htmlspecialchars($store_header['store_name']); ?>
    </div>
    <div style="border-top: 2px solid #000; margin: 5px 0;"></div>
    <div style="border-top: 2px solid #000; margin: 2px 0;"></div>
    <?php
    // Show unpaid credit items for this customer (if any)
    if (!empty($sale['customer_id'])) {
        // Check if unpaid credit section is enabled in settings
        $stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'enable_unpaid_credit_section' AND setting_group = 'receipt_settings'");
        $stmt->execute();
        $enableUnpaidCreditSection = $stmt->fetchColumn();
        
        // Default to enabled (1) if setting doesn't exist
        if ($enableUnpaidCreditSection === false) {
            $enableUnpaidCreditSection = '1';
        }
        
        // Only show unpaid credit section if it's enabled
        if ($enableUnpaidCreditSection === '1') {
            $stmt = $pdo->prepare("
                SELECT 
                    ct.*,
                    s.invoice_number as invoice_no,
                    s.created_at as date,
                    c.name as customer_name,
                    COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as paid,
                    COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as waived,
                    ct.amount - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - 
                    COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as balance,
                    CASE 
                        WHEN COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) + 
                             COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) >= ct.amount THEN 'Paid'
                        WHEN COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) + 
                             COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) > 0 THEN 'Partially Paid'
                        ELSE 'Pending'
                    END as status,
                    s.id as sale_id
                FROM credit_transactions ct 
                JOIN sales s ON ct.sale_id = s.id 
                JOIN customers c ON ct.customer_id = c.id
                LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
                WHERE ct.customer_id = ? 
                GROUP BY ct.id 
                HAVING balance > 0 
                AND (
                    CASE 
                        WHEN COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) + 
                             COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) >= ct.amount THEN 'Paid'
                        WHEN COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) + 
                             COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) > 0 THEN 'Partially Paid'
                        ELSE 'Pending'
                    END IN ('Partially Paid', 'Pending')
                )
                ORDER BY s.created_at DESC
            ");
            $stmt->execute([$sale['customer_id']]);
            $unpaid_credits = $stmt->fetchAll();
            if (!empty($unpaid_credits)) {
                echo '<div style="font-weight:bold; text-align:center; margin-bottom:4px;">UNPAID CREDIT ITEMS</div>';
                
                foreach ($unpaid_credits as $credit) {
                    // Get payment details for this sale
                    $stmt = $pdo->prepare("SELECT method, amount FROM sale_payments WHERE sale_id = ?");
                    $stmt->execute([$credit['sale_id']]);
                    $sale_payments = $stmt->fetchAll();
                    
                    echo '<div style="margin-bottom:8px;">';
                    echo '<div style="font-weight:bold; font-size:11px;">';
                    echo date('M d, Y', strtotime($credit['date'])) . ' - ' . htmlspecialchars($credit['invoice_no']);
                    echo ' (' . htmlspecialchars($credit['status']);
                    if ($credit['waived'] > 0) {
                        echo ' - Waived: ' . $currency . ' ' . number_format($credit['waived'], 2);
                    }
                    echo ')';
                    echo '</div>';
                    
                    // Get items for this sale
                    $stmt = $pdo->prepare("
                        SELECT si.*, p.name as product_name, 
                               GROUP_CONCAT(DISTINCT psn.serial_number ORDER BY psn.created_at ASC) as serial_numbers
                        FROM sale_items si 
                        JOIN products p ON si.product_id = p.id 
                        LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id 
                            AND psn.status = 'sold' 
                            AND psn.source_id = si.sale_id
                            AND psn.source_type = 'credit_sale'
                        WHERE si.sale_id = ?
                        GROUP BY si.id, si.product_id, si.quantity, si.price, si.total, si.discount, si.unit_price, si.total_price, p.name
                    ");
                    $stmt->execute([$credit['sale_id']]);
                    $sale_items = $stmt->fetchAll();
                    
                    // Items table
                    echo '<table style="width:100%; font-size:10px; border-collapse:collapse; margin:2px 0;">';
                    echo '<thead><tr><th style="text-align:left;">Item</th><th style="text-align:center;">Qty</th><th style="text-align:right;">Price</th><th style="text-align:right;">Total</th></tr></thead><tbody>';
                    foreach ($sale_items as $item) {
                        $subtotal = $item['price'] * $item['quantity'];
                        echo '<tr>';
                        echo '<td>' . htmlspecialchars($item['product_name']);
                        if (!empty($item['serial_numbers'])) {
                            $serial_numbers = explode(',', $item['serial_numbers']);
                            echo '<br><small>SN: ' . htmlspecialchars(implode(', ', $serial_numbers)) . '</small>';
                        }
                        echo '</td>';
                        echo '<td style="text-align:center;">' . $item['quantity'] . '</td>';
                        echo '<td style="text-align:right;">' . $currency . ' ' . number_format($item['price'], 2) . '</td>';
                        echo '<td style="text-align:right;">' . $currency . ' ' . number_format($subtotal, 2) . '</td>';
                        echo '</tr>';
                    }
                    echo '</tbody></table>';
                    
                    // Summary (add new Amount and Paid below existing lines)
                    echo '<div style="font-size:11px; text-align:right; margin-top:2px;">';
                    // Existing summary
                    // Calculate cash amount (sum of all non-credit payments)
                    $cash_amount = 0;
                    $payment_methods = [];
                    foreach ($sale_payments as $pay) {
                        if (strtolower($pay['method']) !== 'credit') {
                            $cash_amount += $pay['amount'];
                            $payment_methods[] = ucfirst(strtolower($pay['method']));
                        }
                    }
                    // Calculate total invoice amount
                    $total_invoice = 0;
                    foreach ($sale_items as $item) {
                        $total_invoice += ($item['price'] * $item['quantity']);
                    }
                    echo '<div style="font-weight:bold;">Total Amount: ' . $currency . ' ' . number_format($total_invoice, 2) . '</div>';
                    $paid_line = 'Paid: ' . $currency . ' ' . number_format($cash_amount, 2);
                    if (!empty($payment_methods)) {
                        $paid_line .= ' (' . implode(', ', array_unique($payment_methods)) . ')';
                    }
                    echo '<div>' . $paid_line . '</div>';
                    if ($credit['waived'] > 0) {
                        echo '<div>Waived: ' . $currency . ' ' . number_format($credit['waived'], 2) . '</div>';
                    }
                    // Move Remaining Balance after [Credit Items Table] Paid
                    echo '<div style="font-size:10px; margin-top:2px;">';
                    echo '<span style="font-weight:bold;">[Total Credit Amount] Amount: ' . $currency . ' ' . number_format($credit['amount'], 2) . '</span><br>';
                    echo '<span style="font-weight:bold;">[Credit Amount] Paid: ' . $currency . ' ' . number_format($credit['paid'], 2) . '</span>';
                    echo '<div style="font-weight:bold; color:#e11d48;">Remaining Balance: ' . $currency . ' ' . number_format($credit['balance'], 2) . '</div>';
                    echo '</div>';
                    echo '<div class="dotted-line"></div>';
                }
            }
        }
    }
    ?>
    <?php
    // Gather all receipt info for QR (match main sales receipt content up to thank you)
    $qr_lines = [];
    $qr_lines[] = $store_header['store_name'];
    $qr_lines[] = $store_header['store_address'];
    $qr_lines[] = 'Phone: ' . $store_header['store_phone'];
    $qr_lines[] = 'Email: ' . $store_header['store_email'];
    if (!empty($store_header['website'])) {
        $qr_lines[] = 'Web: ' . $store_header['website'];
    }
    $qr_lines[] = 'Receipt #: ' . $sale['invoice_number'];
    $qr_lines[] = 'Date: ' . $sale['sale_date'];
    $qr_lines[] = 'Client: ' . ($sale['customer_name'] ?? 'Walk-in');
    $qr_lines[] = 'Cashier: ' . ($sale['processed_by'] ?? 'Unknown');
    $qr_lines[] = '---';
    foreach ($sale_items as $item) {
        $item_discount = $item['discount'] ?? 0;
        $subtotal = ($item['price'] * $item['quantity']) - $item_discount;
        $line = $item['product_name'] . ' x' . $item['quantity'] . ' @' . $currency . ' ' . number_format($item['price'], 2) . ' = ' . $currency . ' ' . number_format($subtotal, 2);
        $qr_lines[] = $line;
    }
    $qr_lines[] = '---';
    $qr_lines[] = 'Subtotal: ' . $currency . ' ' . number_format($total, 2);
    if (!empty($sale['shipping']) && $sale['shipping'] > 0) {
        $qr_lines[] = 'Shipping: ' . $currency . ' ' . number_format($sale['shipping'], 2);
    }
    $qr_lines[] = 'Total Payable: ' . $currency . ' ' . number_format($total + $sale['shipping'], 2);
    
    // Add detailed payment breakdown to QR code
    if (count($sale_payments) > 1) {
        $qr_lines[] = 'Payment Breakdown:';
        $actual_paid_amount = 0; // Track only cash/mpesa amounts
        foreach ($sale_payments as $payment) {
            $method = strtolower($payment['method']);
            if ($method === 'cash' || $method === 'mpesa') {
                $actual_paid_amount += $payment['amount'];
            }
            $qr_lines[] = ucfirst($payment['method']) . ': ' . $currency . ' ' . number_format($payment['amount'], 2);
        }
        $qr_lines[] = 'Total Paid: ' . $currency . ' ' . number_format($actual_paid_amount, 2);
    } else {
        // For single payment, check if it's cash/mpesa or credit
        $payment = $sale_payments[0] ?? null;
        if ($payment && (strtolower($payment['method']) === 'cash' || strtolower($payment['method']) === 'mpesa')) {
            $qr_lines[] = 'Amount Paid: ' . $currency . ' ' . number_format($payment['amount'], 2);
        } else {
            $qr_lines[] = 'Amount Paid: ' . $currency . ' ' . number_format(0, 2); // Credit sale, no immediate payment
        }
    }
    
    if (!empty($sale['cash_tendered'])) {
        $qr_lines[] = 'Cash Tendered: ' . $currency . ' ' . number_format($sale['cash_tendered'], 2);
    }
    if (!empty($sale['change_amount'])) {
        $qr_lines[] = 'Change: ' . $currency . ' ' . number_format($sale['change_amount'], 2);
    }
    $qr_lines[] = 'Thank you for shopping with ' . $store_header['store_name'] . '!';
    $qr_data = implode("\n", $qr_lines);
    $qr_url = 'https://api.qrserver.com/v1/create-qr-code/?size=150x150&data=' . urlencode($qr_data);
    ?>
    <div class="qr-section">
      <img src="<?php echo $qr_url; ?>" alt="QR Code" />
    </div>
    <div class="dotted-line"></div>
    <div class="credit-footer">
    <p><i>Designed And Developed By vicotechnologies.co.ke<br> +254702214793</i></p>
    </div>
</div>
</body>
</html> 